// 
// Spark Hott pour télémesure Graupner Hott
// 
// Version 2.08  pour SparkHott 2
// Juin 2018
//
// Auteur : www.thyzoon.fr 
// 2016-2018
// 

//  Thisbip_alarme program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
    
//****************************************************//    
//--                 PIN Mapping                    --//
//                                                    //
// D0 : RXD Réception Hott
// D1 : TXD Emission Hott
// D3 : Entrée signal capteur du compte-tours moteur
// A0 : Tension accu 1 
// A1 : Tension accu 2
// A2 : Tension mesure temperature 1 (RCTN)
// A3 : Tension mesure temperature 2 (RCTN)
//****************************************************// 
    
#include <EEPROM.h>
#include "Message.h"
#include "Mesure.h"

// ---------------
// Fonctions init interruptions
void Init_ctr(); // Initialisation des interruptions compte tours

word filtre(word t); // filtre pour compte-tours

// Def LED verte carte Arduino mini pro (D13)
#define LEDPIN_PINMODE        pinMode (13, OUTPUT);
#define LEDPIN_SWITCH         PINB |= 1<<5;     //switch LEDPIN state (digital PIN 13)
#define LEDPIN_OFF            PORTB &= ~(1<<5);
#define LEDPIN_ON             PORTB |= (1<<5);

// Son des alarmes 
#define ALARME_CARBURANT        0x15  // code son Hott
#define ALARME_TEMPERATURE      0x08  // code son Hott
#define ALARME_TENSION_1_MIN    0x12  // code son Hott
#define ALARME_TENSION_2_MIN    0x13  // code son Hott
#define ALARME_RPM_MAXI         0x19  // code son Hott
#define DUREE_MESSAGE_VOCAL 	5000  // durée d'un message vocal en ms

// Baudrate UART pour Graupner Hott
#define SERIAL_COM_SPEED 19200

// ++++++++++++++++++++++++++++++++
// Variables parametres EEPROM
//
extern word tension1_mini;      // Tension mini accu 1  (format x100 : 7.20V = 720)
extern word tension2_mini;      // Tension mini accu 1        
extern word regime_moteur_mini; // Régime moteur maxi du compte-tours  (format x10 : 11000 trm = 1100)
extern word regime_moteur_maxi; // Régime moteur maxi du compte-tours
extern word chrono_moteur;      // temps alarme moteur // en secondes
extern word alarme_repet;       // temps de repetition alarme // en secondes
extern word temperature_maxi;   // temperature d'alarme moteur // en °C
extern word alarme_on_temperature; // alarme ON temperature
extern word alarme_on_chrono;    // alarme ON chrono temps moteur
extern word alarme_on_v1;        // alarme ON tension V1
extern word alarme_on_v2;        // alarme ON tension V2 

word cpt_alarme;

// ++++++++++++++++++++++++++++++++
// Gestion état du moteur et comptage temps moteur en marche
boolean moteur_marche;        // moteur en marche
unsigned long Compteur_temps_moteur; // en ms
word Temps_moteur;                   // en s 
unsigned long Temps_courant;         // temp en ms
unsigned long Temps_marche_moteur;   // temps  en ms

// ++++++++++++++++++++++++++++++++
// Alarmes
byte bip_alarme;                     // son de l'alarme Hott
byte derniere_alarme;                // derniere alarme envoyée (son de l'alarme Hott)
boolean flag_alarm_temps_moteur;     // Alarme carburant (temps moteur atteint)
boolean flag_alarm_v1;               // Alarme tension accu 1
boolean flag_alarm_v2;               // Alarme tension accu 2
boolean flag_alarm_temperature;      // Alarme temperature moteur
boolean flag_alarm_rpm_max;          // Alarme regime moteur trop haut
word temps_alarme;                   // temps prochaine alarme
int Temperature_capteur_1;             // Temperature 1 du moteur 
int Temperature_capteur_2;             // Temperature 2 du moteur 
unsigned long Temps_derniere_alarme; // temp en ms

// ++++++++++++++++++++++++++++++++
// Mesure compte-tours

volatile unsigned long temps_helice;  // temps passages de pales en µs
volatile boolean flag_temps_helice;   // nouveau temps hélice valide (interruption traitee)
word trmn_helice;                     // tours par minutes mesure
word trmn_helice_valide;              // tours par minutes valide
word trmn_helice_hott;                // tours par minutes vers telemesure Hott
word delta_trmn_helice;               // ecart entre 2 mesure / tours par minutes
byte ct_ctr_valide;                   // comptage compte-tours valide

// Pour detection de moteur à l'arrêt
unsigned long tps_a_ms;        // temps en ms
unsigned long tps_b_ms;        // temps en ms

// Gestion messagerie Hott
GMessage message;

// Gestion des mesure de tension
Mesure analogique;

//********************************************
// Initialisations 
//
void setup() {
  // Init communications 
  message.init();
  // Init mesures analogiques
  analogique.init();
  
  // LED carte
  LEDPIN_PINMODE
  LEDPIN_OFF 
  
  // Init variables compte-tours
  trmn_helice = 0;
  trmn_helice_valide = 0;  
  trmn_helice_hott = 0;
  delta_trmn_helice = 0;
  tps_a_ms =0;
  tps_b_ms =0;
  ct_ctr_valide = 0;       // comptage compte-tours valide  
  moteur_marche = false;
 
  // Init variables temps moteur en fonctionnement
  Compteur_temps_moteur = 0;
  Temps_moteur = 0; 
  Temps_courant = 0;  
  Temps_marche_moteur = 0; 
  
  // Gestion alarmes
  flag_alarm_temps_moteur = false;
  flag_alarm_v1 = false;               
  flag_alarm_v2 = false;      
  flag_alarm_temperature = false;  
  flag_alarm_rpm_max = false; 
  temps_alarme = chrono_moteur; // temps pour la premiere alarme moteur
  bip_alarme = 0;       // pas de son
  cpt_alarme = 0; 
  Temps_derniere_alarme = 0;
  derniere_alarme = 0;
   
  // Init entrée compte-tours moteur et interruptions  
  flag_temps_helice = false;
  temps_helice = 0; 
  Init_ctr ();   
}

//******************************************
// Boucle principale
//
void loop() {   

  // ---------------------------------------------
  // Régime moteur
  // Traitement temps entre détection de pales d'hélice (capteur magnetique)
  //
  if (flag_temps_helice == true) {    // Vrai si mesure temps en interruption
    LEDPIN_ON                         // LED carte arduino  ON
    flag_temps_helice = false;	      // acquitement interruption
    // -->   calcul tours / minutes
    trmn_helice = 60000000 / temps_helice;       // 60 (s) x 10e6 = 60000000 (60s par minutes et temps_helice en µs) 
	  
    // Traitement de la mesure : 
    // Fenetre MIN / MAX , variation maxi de regime -->   filtrage / MAX = regime max + 2000 tr/m
    if ((trmn_helice >= regime_moteur_mini * 10) && (trmn_helice <= ((regime_moteur_maxi*10)+2000))) {        
      // Filtrage
      trmn_helice_valide = filtre (trmn_helice);      // Filtrage compte-tours  	  
    }	  
    else  // valeur non conforme
      trmn_helice_valide = 0;

    //-------------------
    // Moteur en marche ?    
    if (trmn_helice_valide > 0){
      tps_a_ms = millis();        // temps courant moteur tournant
      if (ct_ctr_valide > 4) { // 5 valeurs valides : par exemple à 1500tr/min  => temps = 200ms.      
        trmn_helice_hott = trmn_helice_valide;   // valeur compte-tours pour telemesure Hott
        moteur_marche = true;                    // le moteur tourne !      
      }  
      else {           
        ct_ctr_valide ++;
      }
    }
    LEDPIN_OFF                  // LED carte arduino  OFF
  }
  else { // pas de valeur valide (trm_helice)
    // Detection d'arret moteur
    tps_b_ms = millis(); // temps courant
    if ((tps_b_ms - tps_a_ms) > 200){   // 200 ms sans "interruption temps hélice" (300 tr/min) = arret moteur detecté 
      trmn_helice_hott = 0;    // valeur compte-tours pour telemesure Hott = 0
      ct_ctr_valide = 0;       // reset	compteur  
      moteur_marche = false;   // arret moteur
    } 
  }  
  
  // Detection sur-regime moteur : Alarme
  if (trmn_helice_hott > (regime_moteur_maxi*10))
    flag_alarm_rpm_max = true;    
  
  //---------------------------------------------------------------------------------------
  // Gestion du chronometre temps moteur pour gestion de l'alarme sur temps moteur tournant   
  Temps_courant = millis();  // temps courant ms
  if (moteur_marche == true) {
    // Calcul du temps moteur tournant par cumul des temps 
    Compteur_temps_moteur = Compteur_temps_moteur + (Temps_courant - Temps_marche_moteur);
    Temps_moteur = Compteur_temps_moteur / 1000; //  en secondes  
  }
  Temps_marche_moteur = Temps_courant;    // memorise le temps courant pour le prochain calcul 
   
  //-----------------------------------
  // Alarme depassement du temps moteur 
  // Temps maxi atteint 
  if (alarme_on_chrono)
    if (Temps_moteur > chrono_moteur) {
      chrono_moteur = chrono_moteur + alarme_repet; // nouveau temps = temps precedent + temps de repetition
      flag_alarm_temps_moteur = true;
    }   
  
  //-----------------------------------
  // Gestion des messages Hott / Menus et telemesure
  // 
  // Calcul de la température
  // Le temps de calcul de la température étant assez long,
  // il n'est effectué qu'après chaque envoie de trame Hott. (Periode de temps libre)
  // Ceci permet de synchroniser le cacul sur la cadence des trames. 
  // Idem pour les 2 mesures de tension Accu 1 et Accu 2
  if (message.main()) {
    bip_alarme = 0;   // remise à zero de l'alarme après une émission
    
    Temperature_capteur_1 = analogique.Temperature (1); // mesure temperature 
    Temperature_capteur_2 = analogique.Temperature (2); // mesure temperature 
	
    if (alarme_on_temperature){
      // Alarme temperature moteur    
      if  ((Temperature_capteur_1 > temperature_maxi) || (Temperature_capteur_2 > temperature_maxi)) 
      flag_alarm_temperature = true;   
    }
    
  //-----------------------------------  
  // Alarme tension accu 1
  // detection tension sous le seuil configuré
  if (alarme_on_v1)
    if (analogique.Tension_LI1 () < ((float) tension1_mini / 100.0))
      flag_alarm_v1 = true;
    else
      flag_alarm_v1 = false;  

  //-----------------------------------
  // Alarme tension accu 2
  if (alarme_on_v2)  
    if (analogique.Tension_LI2 () < ((float) tension2_mini / 100.0))  
      flag_alarm_v2 = true;
    else
      flag_alarm_v2 = false;          
  }    ///---- fin 'if (message.main())'
  
  //----------------------------------------
  // Gestion de priorité des alarmes
  // Les radios MX-xx ne traitent qu'une alarme vocale à la fois,
  // Si l'on envoie plusieurs alarmes à la suite sans attendre la
  // fin du message vocal en cours, les alarmes suivantes sont perdues.
  //
  // L'alarme température moteur est traitée en priorité car elle peut occassionner un calage moteur
  //   
  // L'alarme carburant (temps de vol) est traité en seconde priorité et elle n'est pas
  // bloquante car elle passe seulement toutes les 15s, 30s 45s ou 60 s selon configuration
  // 
  // Les alarmes "tension batterie" sont traitées à la suite. 
  //   (hypothèse: redondance 2 batteries donc moins critique)
  //
  // Il est normalement peu probable que toutes les alarmes soient simultanées.
  // La derniere alarme est mémorisée afin qu'elle ne passe pas 2 fois si une autre alarme est positionnée
  // Ce système ne permet de passer que 2 alarmes, s'il en a une 3ieme, elle est masquée.
  //
 
  // Gestion du temps entre alarme afin de laisser assez de temps au message 
  // vocal d'être émis complètement avant envoi de l'alarme suivante
  Temps_courant = millis();  // temps courant ms
  if ((bip_alarme == 0) && (Temps_courant > (Temps_derniere_alarme + DUREE_MESSAGE_VOCAL))) {
    Temps_derniere_alarme = Temps_courant;

    //--------------------------------
    // prise en compte des flags alarme 
    // Ordre inverse de la priorité
    //
    if ((flag_alarm_rpm_max)&&(derniere_alarme!=ALARME_RPM_MAXI)) {   
	  flag_alarm_rpm_max = false;    // prise en compte
	  bip_alarme = ALARME_RPM_MAXI;  // son alarme regime moteur
    }
    
    if ((flag_alarm_v2)&&(derniere_alarme!=ALARME_TENSION_2_MIN)) {   
	  flag_alarm_v2 = false;              // prise en compte
	  bip_alarme = ALARME_TENSION_2_MIN;  // son alarme tension min 2
    }

    if ((flag_alarm_v1)&&(derniere_alarme!=ALARME_TENSION_1_MIN)){   
	  flag_alarm_v1 = false;              // prise en compte
	  bip_alarme = ALARME_TENSION_1_MIN;  // son alarme tension min 1
    } 
    
    if ((flag_alarm_temps_moteur)&&(derniere_alarme!=ALARME_CARBURANT)) {     // Alarme carburant
      flag_alarm_temps_moteur = false;      // prise en compte
      bip_alarme = ALARME_CARBURANT;        // Son ou message vocal
    }
	
    if ((flag_alarm_temperature)&&(derniere_alarme!=ALARME_TEMPERATURE))  {          // Alarme température
  	  flag_alarm_temperature = false;   // prise en compte
	  bip_alarme = ALARME_TEMPERATURE;  // son alarme temperature ou message
    }
    
    if (bip_alarme)
      derniere_alarme = bip_alarme;// memorise la dernière alarme.
    else
      derniere_alarme = 0;
  }
}
